//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Dallas Semiconductor
// shall not be used except as stated in the Dallas Semiconductor
// Branding Policy.
//---------------------------------------------------------------------------
//
//  PRINTTAG.C   - source file containing the Tag print functions using
//                 Object-Based Messaging format.
//
//  Version: 2.00
//  Revisions:
//

#define TAG_MSGS

// includes
#include "msginc.h"

// Print TAG functions
int TAGToString(TAGType *, char *, int);
int TAGEndToString(int, char *, int);
int TAGDataToString(ushort, uchar *, int, int, char *, int);
int TAGGroupToString(ushort, int, int, char *, int);
int MessageTAGLookUp(ushort, char *);

//--------------------------------------------------------------------------
// Print TAG functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Parse a Tag and print a represntation of it in 'pmsg' with a maximum
// length of 'msgmax'.  The '*Tag' is the tag to parse into a string.
//
// Returns:    TRUE  - Tag parsed and converted to a string
//             FALSE - error parsing or printing
//
int TAGToString(TAGType *TAG, char *pmsg, int msgmax)
{
   ushort OD;
   int i,addlen,msgcnt;

   // peek ahead to make sure this is parse-ble data
   if ((TAG->msg[0] != (GD_ParseData & 0xFF)) ||
       (TAG->msg[1] != (GD_ParseData >> 8))) 
   {
      sprintf(pmsg,"Data is not parse-able\n");
      return FALSE;
   }   

   // start the counters
   TAG->parse = 0;
   TAG->depth = 0;
   pmsg[0] = 0;

   // print the tag as raw data
   msgcnt = sprintf(pmsg,"RAW TAG: (size %d)\n",TAG->msglen);
   for (i = 0; i < TAG->msglen; i++)
   {
      msgcnt += sprintf(&pmsg[msgcnt],"%02X ",TAG->msg[i]);
      if ((i != 0) && (((i + 1) % 16) == 0))
         msgcnt += sprintf(&pmsg[msgcnt],"\n");
   }
   msgcnt += sprintf(&pmsg[msgcnt],"\n\n");

   // loop to parse the message
   while (TAG->parse < TAG->msglen)
   {
      // check for an end of group
      if (TAG->msg[TAG->parse] == 0xFF)
      {
         addlen = TAGEndToString(TAG->depth - 1, pmsg + msgcnt, msgmax - msgcnt);
         if (!addlen)
            return FALSE;
         msgcnt += addlen;
         TAG->depth--;
         TAG->parse++;
         continue;
      }
   
      // check if have an OD left if buffer
      if ((TAG->parse + 2) > TAG->msglen)
         break;

      // get an OD
      OD = TAG->msg[TAG->parse++];
      OD |= (ushort)(TAG->msg[TAG->parse++] << 8);

      // print either a group or data object
      if (OD & GROUP)
      {
         addlen = TAGGroupToString(OD, TAG->depth, FALSE, pmsg + msgcnt, msgmax - msgcnt);
         if (!addlen)
            return FALSE;
         msgcnt += addlen;
         TAG->depth++;
      }
      else
      {
         addlen = TAGDataToString(OD, &TAG->msg[TAG->parse], TAG->depth, FALSE, 
                                  pmsg + msgcnt, msgmax - msgcnt);
         if (!addlen)
            return FALSE;
         msgcnt += addlen;
         TAG->parse += (short)(TAG->msg[TAG->parse] + 1);
      }
   }

   return TRUE;
}

//--------------------------------------------------------------------------
// Add a string respresentation of a the specified data object 'OD' to
// the buffer 'pmsg' with a max length of 'msgmax'.  A pointer to the
// objects payload is also passed '*payload'.  The 'depth' is used
// for printing the correct indentation.  The flag 'simple' is used
// to print the full OBM format (FALSE) or a condensed format (TRUE).
//
// Returns:    >0    - message added, return number of characters
//             FALSE - error parsing or not enough room in msg buffer
//
int TAGDataToString(ushort OD, uchar *payload, int depth, int simple,
                    char *pmsg, int msgmax)
{
   char lumsg[256],msg[PARSEMAX];
   int msgcnt=0,i,len,l,pcnt=0;
   ulong ul;

   // zero the string
   pmsg[0] = 0;

   // check for NULL payload
   if (payload == NULL)
      return 0;

   // indent number of depth
   for (i = 0; i < depth; i++)
      msgcnt += sprintf(msg + msgcnt,"   ");

   // get the length
   len = payload[pcnt++];

   // check for zero length payload
   if (len == 0)
      return 0;

   // lookup the group
   MessageTAGLookUp(OD,lumsg);
   if (simple)
      msgcnt += sprintf(msg + msgcnt,"%s: ",lumsg);
   else
      msgcnt += sprintf(msg + msgcnt,"{D: %s(%d)}",lumsg,len);

   // check of zero length
   if (len != 0)
   {
      // case depending on Data Object Descriptor
      switch (OD & DOD_MASK)
      {
         case DOD_NBO_Int:  // NBO decimal integer
            // loop to find out how many characters I need to send this int
            ul = 0;
            for (l = len-1; l >= 0; l--)
               ul |= (payload[pcnt++] << (8 * l));
            // add to the message
            msgcnt += sprintf(msg + msgcnt," %d",ul);
            break;
         case DOD_TimeDate:  // Time / Date   
         case DOD_Integer:  // decimal numeric value, leading zero supressed
            // loop to find out how many characters I need to send this int
            ul = 0;
            for (l = 0; l < len; l++)
               ul |= (payload[pcnt++] << (8 * l));
            // add to the message
            msgcnt += sprintf(msg + msgcnt," %d",ul);
            break;
         case DOD_DSROM:  // Dallas Serial ROM number uncompressed
            msgcnt += sprintf(msg + msgcnt," ");
            // loop for all rom bytes
            for (i = (len - 1); i >= 0; i--)
               msgcnt += sprintf(msg + msgcnt,"%02X",payload[pcnt + i]);
            break;
         case DOD_String:  // printable ASCII string
            msgcnt += sprintf(msg + msgcnt," ");
            // loop for all string chars
            for (i = 0; i < len; i++)
               msgcnt += sprintf(msg + msgcnt,"%c",payload[pcnt++]);
            break;
         case DOD_DSROM_Comp:    // Dallas Serial ROM number compressed (nocrc)
         case DOD_DotDec:        // Dotted-decimal notation (as in IP addresses) NBO
         case DOD_ColHex:        // Colon delimited Hex (as in eithernet address) NBO
         case DOD_Boolean:       // flags
         case DOD_Unknown:       // Unknown format
         case DOD_Password:      // password (not displayed or masked by * characters
         case DOD_ByteArray:     // Byte array of hex digits
         default:
            if ((msgcnt + len) > PARSEMAX) 
               msgcnt += sprintf(msg + msgcnt,"TOO LONG");
            else
            {
               // loop for all bytes
               for (i = 0; i < len; i++)
                  msgcnt += sprintf(msg + msgcnt," %02X",payload[pcnt++]);
               break;
            }
      }
   }

   // add crlf
   msgcnt += sprintf(msg + msgcnt,"\n");

   // check if enough room
   if (msgcnt < msgmax)
   {
      for (i = 0; i < msgcnt; i++)
         pmsg[i] = msg[i];
      pmsg[i] = 0;
      return msgcnt;
   }
   else
      return FALSE;
}

//--------------------------------------------------------------------------
// Add a string respresentation of a the specified group object 'OD' to
// the buffer 'pmsg' with a max length of 'msgmax'.  The 'depth' is used
// for printing the correct indentation.  The flag 'simple' is used
// to print the full OBM format (FALSE) or a condensed format (TRUE).
//
// Returns:    >0    - message added, return number of characters
//             FALSE - error parsing or not enough room in msg buffer 
////
int TAGGroupToString(ushort OD, int depth, int simple, char *pmsg, int msgmax)
{
   char lumsg[80],msg[200];
   int msgcnt=0,i;

   // indent number of depth
   for (i = 0; i < depth; i++)
      msgcnt += sprintf(msg + msgcnt,"   ");
      
   // lookup the group
   MessageTAGLookUp(OD,lumsg);
   if (simple)
      msgcnt += sprintf(msg + msgcnt,"%s: \n",lumsg);
   else
      msgcnt += sprintf(msg + msgcnt,"{G: %s}[\n",lumsg);

   // check if enough room
   if (msgcnt < msgmax)
   {
      for (i = 0; i < msgcnt; i++)
         pmsg[i] = msg[i];
      pmsg[i] = 0;
      return msgcnt;
   }
   else
      return FALSE;
}

//--------------------------------------------------------------------------
// Add a string respresentation of the OBM end group symbol at the 
// specified 'depth'.
//
// Returns:    >0    - message added, return number of characters
//             FALSE - error parsing or not enough room in msg buffer 
//
int TAGEndToString(int depth, char *pmsg, int msgmax)
{
   char msg[200];
   int msgcnt=0,i;

   // indent number of depth
   for (i = 0; i < depth; i++)
      msgcnt += sprintf(msg + msgcnt,"   ");
      
   // print end group marker
   msgcnt += sprintf(msg + msgcnt,"]\n");

   // check if enough room
   if (msgcnt < msgmax)
   {
      for (i = 0; i < msgcnt; i++)
         pmsg[i] = msg[i];
      pmsg[i] = 0;
      return msgcnt;
   }
   else
      return FALSE;
}

//--------------------------------------------------------------------------
// Copy the string that represents the object OD into the provided buffer
// 'pmsg'. 
//
// Returns:    TRUE  - string representation found and added.
//             FALSE - string representation not found.
//
int MessageTAGLookUp(ushort OD, char *pmsg)
{
   int cnt,pcnt=0;
   ushort TOD;

   // loop to find and exact match
   cnt = 0;
   do
   {
      if (TAGMsgs[cnt].val == OD)
      {
         sprintf(pmsg,TAGMsgs[cnt].msg);
         return TRUE;
      }
      cnt++;
   }
   while (TAGMsgs[cnt].msg[0] != 0);

   // Not standard, look up the parts DOD, DOM, GOD ...
   pcnt = sprintf(pmsg,"no lookup #%04X",OD);
   // loop to find and DOD
   TOD = (ushort)(OD & DOD_MASK);
   cnt = 0;
   do
   {
      if (TAGMsgs[cnt].val == TOD)
      {
         pcnt += sprintf(&pmsg[pcnt],", %s",TAGMsgs[cnt].msg);
         break;
      }
      cnt++;
   }
   while (TAGMsgs[cnt].msg[0] != 0);
   // loop to find and DOM
   TOD = (ushort)(OD & DOM_MASK);
   cnt = 0;
   do
   {
      if (TAGMsgs[cnt].val == TOD)
      {
         pcnt += sprintf(&pmsg[pcnt],", %s",TAGMsgs[cnt].msg);
         break;
      }
      cnt++;
   }
   while (TAGMsgs[cnt].msg[0] != 0);
   // loop to find and GOD
   TOD = (ushort)(OD & GOD_MASK);
   cnt = 0;
   do
   {
      if (TAGMsgs[cnt].val == TOD)
      {
         pcnt += sprintf(&pmsg[pcnt],", %s",TAGMsgs[cnt].msg);
         break;
      }
      cnt++;
   }
   while (TAGMsgs[cnt].msg[0] != 0);

   return FALSE;
}

