//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  1wsnsut.C - 1-Wire Sensor utility functions.  Run scripts to perform
//              1-Wire sensor operations.  Use variables in MIB as needed.
//
//  Version:   2.00
//  History:   
//

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "ownet.h"
#include "1wsensor.h"   
#include "objects.h"
#include "tagconst.h"
#include "swt12.h"

// external function prototypes
extern int ReadFromMIB(ushort, uchar *, int);
extern void output_status(int, char *);
extern void msDelay(int);
extern int owAccess(int);
extern int owTouchByte(int, int);
extern int owVerify(int, int);
extern int ReadTemperature(int,uchar *,float *);
extern void owSerialNum(int, uchar *, int);
extern short ReadSwitch12(int,int);
extern int owFirst(int,int,int);
extern int owNext(int,int,int);
extern ushort docrc16(int,ushort);
extern int bitacc(int,int,int,uchar *);
extern int ReadCounter(int,uchar *,int,ulong *);
extern int owBlock(int,int,uchar *,int);
extern int SetSwitch12(int,uchar *,SwitchProps);
extern int SetSwitch1F(int,uchar *,int,int,uchar *,int);
extern int owReadFile(int,uchar *,uchar *);
extern void owFamilySearchSetup(int,int);
extern long msGettick(void);
extern int key_abort(void);

// Local Function Prototypes 
int RunScript(int, Script script[], void *, int);
int ScriptStep(int, Script *, int *, int *, int *, char *, void *, int);
int DeviceSetVerify(int, uchar *);
int Read1820Temp(float *);
int Set2406State(int portnum, uchar *);
int Read2423Counter(int,uchar *,ulong *);
int ReadTagNext(int, int *, uchar *, int);
int ROMToStr(char *, uchar *);
int ROMSwitchState(int,uchar *);
int DeviceCheckedForTag(int,uchar **);
int Set2409State(int portnum, uchar *);
int Read2409Info(int,uchar *);

// defines
#define ROM_STACK_MAX   200
#define RETRY_MAX       5

// globals
static uchar rom_stack[ROM_STACK_MAX][8];
static int stack_num=0;
static uchar retry_stack[ROM_STACK_MAX];
static uchar StepSN[8];

//--------------------------------------------------------------------------
// Script functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Run the specified script. 
// Status is printed using 'output_status' function.  The script result is
// returned in the buffer 'prslt' of max length 'rslt_max'.  Result data 
// format is script dependent.
//
// Return: TRUE - all steps completed
//         FALSE - all steps NOT completed due to error or escape 
//
int RunScript(int portnum, Script script[], void *prslt, int rslt_max)
{
   char statusmsg[256],LastDescription[256],LastMsg[256];
   char tmsg[256];
   int StepCount,SubStep,ErrorCount,Status,i;
   uchar *puchar;   

   // reset the step to the begining
   StepCount = 0;
   SubStep = 0;
   ErrorCount = 0;
   Status = STATUS_INPROGRESS;
   statusmsg[0] = 0;
   puchar = prslt;
   for (i = 0; i < rslt_max; i++)
      puchar[i] = 0;
 
   // loop to perform all of the steps to download the Thermochron
   do
   {   
      // switch on the status of the last step done
      switch(Status)
      {
         // step complete so go to the next
         case STATUS_STEP_COMPLETE:
            StepCount++;
            SubStep = 0;
            ErrorCount = 0;
            Status = STATUS_INPROGRESS;
            LastDescription[0] = 0;
            LastMsg[0] = 0;
            break;
         // in progress so call again
         case STATUS_INPROGRESS:
            // print step description if different 
            if (strcmp(LastDescription,
                script[StepCount].StepDescription) != 0)
            {
               sprintf(tmsg,"%s --> ",script[StepCount].StepDescription);
               output_status(LV_VERBOSE,tmsg);
               sprintf(LastDescription,"%s",script[StepCount].StepDescription);
            }

            // perform a step in the job
            ScriptStep(portnum, &script[StepCount],&SubStep, 
                      &Status, &ErrorCount, statusmsg, 
                      prslt, rslt_max);

            // print results if different
            if (strcmp(LastMsg,statusmsg) != 0)
            {
               sprintf(tmsg,"%s\n",statusmsg);
               output_status(LV_VERBOSE,tmsg);
               sprintf(LastMsg,"%s",statusmsg);
            }
            else
               output_status(LV_VERBOSE,".");
            break;
         // encountered a transient error
         case STATUS_ERROR_TRANSIENT:
            // if 20 tansient errors in a row then abort
            if (ErrorCount > 20)
               Status = STATUS_ERROR_HALT;
            else
               Status = STATUS_INPROGRESS;
            break;
         // all steps complete
         case STATUS_COMPLETE:
            output_status(LV_VERBOSE,"End script normally\n");
            return TRUE;
         // non-recoverable error
         case STATUS_ERROR_HALT:
            // try to do emergency clean up
            output_status(LV_ALWAYS,"Aborting script due to non-recoverable error\n");
            return FALSE;
      }
   }
   while (!key_abort());

   // key abort
   output_status(LV_ALWAYS,"Aborting script due to key press\n");
   return FALSE;
}

//----------------------------------------------------------------------
//  Use the script to perform a step and return.
//
//  Return:
//          STATUS_STEP_COMPLETE - step complete, go to next step
//          STATUS_INPROGRESS - step is not complete, call again
//          STATUS_ERROR_HALT -  error, halt script
//          STATUS_ERROR_TRANSIENT - transient error, call again
//
int ScriptStep(int portnum, Script *StateScript, int *SubStep,
              int *Status, int *ErrorCount, char *statusmsg,
              void *prslt, int rslt_max)
{
   int *pint;
   float tmp, *pfloat;
   uchar tbuf[255],*puchar;
   int scnt,obln,state,i;   
   ulong tulong, *pulong;
   uchar trom[8];
   uchar infob;
   short info;

   // do the current step and return
   switch (StateScript->Step)
   {
      // attempt to get a session on the 1-Wire Net
      case ST_GET_SESSION:
         *Status = STATUS_STEP_COMPLETE;
         break;
      
      // perform TMSetup on the 1-Wire Net port   
      case ST_SETUP:
         *Status = STATUS_STEP_COMPLETE;
         break;

      // release the session on the 1-Wire Net
      case ST_REL_SESSION:
         *Status = STATUS_STEP_COMPLETE;
         break;      

      // the operation is complete      
      case ST_FINISH:
         sprintf(statusmsg,"Operation complete");
         *Status = STATUS_COMPLETE;
         break;      

      // Select and verify device
      case ST_VERIFY:
         // get the rom from the mib and set
         if (DeviceSetVerify(portnum, &StepSN[0]))
         {
            for(i=7; i>=0; i--)
               sprintf(statusmsg, "%02X", StepSN[i]);

            sprintf(statusmsg," Device found and selected");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         { 
            sprintf(statusmsg," Error, could not find device");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;      

      //Recall EE calibration
      case ST_EE_RECALL:
         // step not used now, remove next version
         *Status = STATUS_STEP_COMPLETE;
         break;      

      // Start Temperation Conversion
      case ST_TEMP_CONV_1820:
         *Status = STATUS_STEP_COMPLETE;
         break;      

      // Delay 100ms
      case ST_DELAY_100MS:
         if(owVerify(portnum, FALSE))
         {
            msDelay(100);
            sprintf(statusmsg, "Session valid, delay 100ms complete");
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         {
            sprintf(statusmsg, "Error, session not valid.");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_HALT;
         }
         break;      
            
      // Read the temperature
      case ST_READ_TEMP_1820:
         if (ReadTemperature(portnum, &StepSN[0], &tmp))
         {
            // set to return value 
            pfloat = prslt;
            *pfloat = tmp;

            sprintf(statusmsg,"Temperature read success: "
                              "%5.1f C, %5.1f F\n",tmp,
                               tmp * 9 / 5 + 32);
            *Status = STATUS_STEP_COMPLETE;
         }
         else
         { 
            sprintf(statusmsg,"Error, could not read temperature");
            *Status = STATUS_ERROR_HALT;
         }
         break;      

      // Set the branch state
      case ST_SET_BRANCH_2406:
         if (Set2406State(portnum, &StepSN[0]))
         {
            sprintf(statusmsg,"Branch state set");
            *Status = STATUS_STEP_COMPLETE;
            pint = prslt;
            *pint = TRUE;
         }
         else
         { 
            sprintf(statusmsg,"Error, setting branch state");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
            pint = prslt;
            *pint = FALSE;
         }
         break;

      // Set the branch state
      case ST_SET_BRANCH_2409:
         if (Set2409State(portnum, &StepSN[0]))
         {
            sprintf(statusmsg,"Branch state set");
            *Status = STATUS_STEP_COMPLETE;
            pint = prslt;
            *pint = TRUE;
         }
         else
         { 
            sprintf(statusmsg,"Error, setting branch state");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
            pint = prslt;
            *pint = FALSE;
         }
         break;      

      // verify the state of a rom switches
      case ST_VERIFY_SWITCH:
         // read the MIB for the array or ROM switches
         obln = ReadFromMIB(DD_InitState, &tbuf[0], sizeof(tbuf));
         if (obln <= 0)
         { 
            sprintf(statusmsg,"Error, could not read init buffer in MIB");
            *Status = STATUS_ERROR_HALT;
         }
         
         // use SubStep to see which one to look at
         if (*SubStep >= (obln / 8))
         {
            sprintf(statusmsg,"Last ROM switch checked");
            *Status = STATUS_STEP_COMPLETE;
         }
         else 
         {
            // print rom and then the state
            scnt = ROMToStr(&statusmsg[0],&tbuf[*SubStep * 8]);

            state = ROMSwitchState(portnum,&tbuf[*SubStep * 8]);
            if (state)
               sprintf(&statusmsg[scnt]," switch ON");
            else
               sprintf(&statusmsg[scnt]," switch OFF");
                  
            // collect array of bits for state return
            if (*SubStep < (rslt_max * 8))
               bitacc(WRITE_FUNCTION, state, *SubStep, (uchar *)prslt);   
               
            // increment to look at the next
            *SubStep = *SubStep + 1;
            *Status = STATUS_INPROGRESS;
         }
         break;

      // read the counter specified by the MIB
      case ST_READ_COUNT_2423:
         if (Read2423Counter(portnum, &StepSN[0], &tulong))
         {
            sprintf(statusmsg,"Counter read: %d",tulong);
            *Status = STATUS_STEP_COMPLETE;
            pulong = prslt;

            if (pulong[0] == 0)
            {
               pulong[0] = msGettick();
               pulong[1] = tulong;
            }
            // second and final reading
            else
            {
               pulong[2] = msGettick();
               pulong[3] = tulong;
            }
         }
         else
         { 
            sprintf(statusmsg,"Error, reading counter value");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;      

      // find the next device and attempt to get a TAG
      case ST_READ_TAG_NEXT:
         *Status = ReadTagNext(portnum,SubStep,prslt,rslt_max);
         if (*Status == STATUS_STEP_COMPLETE)
         {
            // check prslt to see if tag data is in it
            puchar = prslt;
            if ((puchar[0] != 0) && (puchar[1] != 0))
            {
               scnt = sprintf(statusmsg,"Tag file found on ");
               // print rom
               owSerialNum(portnum, &(uchar)trom[0], TRUE);
               for (i = 7; i >= 0; i--)
                  scnt += sprintf(statusmsg + scnt,"%02X",trom[i]);
               scnt += sprintf(statusmsg + scnt,"\n");
               output_status(LV_ALWAYS,statusmsg);
            }
            scnt = sprintf(statusmsg,"Tag search complete ");
         }
         else if (*Status == STATUS_ERROR_TRANSIENT)
         {
            scnt = sprintf(statusmsg,"Error reading Tag");
            *ErrorCount = *ErrorCount + 1;
         }
         else if (*Status == STATUS_INPROGRESS)
            sprintf(statusmsg,"Interum step in tag search complete");
         break;

      // read the level specified by the MIB
      case ST_READ_LEVEL_2406:
         info = ReadSwitch12(portnum, TRUE);
         if (info != -1)
         {
            sprintf(statusmsg,"Switch State read: %02X",info);
            *Status = STATUS_STEP_COMPLETE;
            if (rslt_max >= 2)
            {
               // calculate the state and add the mask
               puchar = prslt;
               // state in byte 1
               puchar[0] = (uchar)((info >> 2) & 0x03);
               // get mask for byte 2 read the mib entry
               ReadFromMIB(DD_ChannelMask, &puchar[1], 1);
            }
            else
            {   
               *Status = STATUS_ERROR_HALT;
               sprintf(statusmsg,"Result length of %d too small.\n",rslt_max);
            }
         }
         else
         { 
            sprintf(statusmsg,"Error, reading switch info");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;      

      // read the level specified by the MIB
      case ST_READ_LEVEL_2409:
         if (Read2409Info(portnum,tbuf))
         {
            sprintf(statusmsg,"Switch State read: %02X",tbuf[0]);
            *Status = STATUS_STEP_COMPLETE;
            if (rslt_max >= 2)
            {
               // calculate the state and add the mask
               puchar = prslt;
               // state in byte 1
               puchar[0] = (uchar)(((tbuf[0] >> 1) & 0x01) |
                                   ((tbuf[0] >> 2) & 0x02));
               // get mask for byte 2 read the mib entry
               ReadFromMIB(DD_ChannelMask, &puchar[1], 1);
            }
            else
            {   
               *Status = STATUS_ERROR_HALT;
               sprintf(statusmsg,"Result length of %d too small.\n",rslt_max);
            }
         }
         else
         { 
            sprintf(statusmsg,"Error, reading switch info");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;      

      // read the activity specified by the MIB
      case ST_READ_ACTIVITY_2406:
         info = ReadSwitch12(portnum, FALSE);
         if (info != -1)
         {
            sprintf(statusmsg,"Raw switch state read: %02X",info);
            *Status = STATUS_STEP_COMPLETE;
            if (rslt_max >= 2)
            {
               // calculate the state and add the mask
               puchar = prslt;
               // activity state in byte 1
               puchar[0] = (uchar)((info >> 4) & 0x03);
               // get mask for byte 2 read the mib entry
               ReadFromMIB(DD_ChannelMask, &puchar[1], 1);
            }
            else
            {   
               *Status = STATUS_ERROR_HALT;
               sprintf(statusmsg,"Result length of %d too small.\n",rslt_max);
            }
            // clear the activity
            info = ReadSwitch12(portnum, TRUE);
         }
         else
         { 
            sprintf(statusmsg,"Error, reading switch info");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;

      // read the activity specified by the MIB
      case ST_READ_ACTIVITY_2409:
         if (Read2409Info(portnum, tbuf))
         {
            sprintf(statusmsg,"Raw switch state read: %02X",tbuf[0]);
            *Status = STATUS_STEP_COMPLETE;
            if (rslt_max >= 2)
            {
               // calculate the state and add the mask
               puchar = prslt;
               // activity state in byte 1
               puchar[0] = (uchar)((tbuf[0] >> 4) & 0x03);
               // get mask for byte 2 read the mib entry
               ReadFromMIB(DD_ChannelMask, &puchar[1], 1);
            }
            else
            {   
               *Status = STATUS_ERROR_HALT;
               sprintf(statusmsg,"Result length of %d too small.\n",rslt_max);
            }
            // clear the activity (send all-lines-off)
            SetSwitch1F(portnum, &StepSN[0], 0, 1, &infob, TRUE);
         }
         else
         { 
            sprintf(statusmsg,"Error, reading switch info");
            *ErrorCount = *ErrorCount + 1;
            *Status = STATUS_ERROR_TRANSIENT;
         }
         break;      

      default:
         *Status = STATUS_ERROR_HALT;
         sprintf(statusmsg,"1-Wire Step not known\n");
         break;
   }

   return *Status;
}

//--------------------------------------------------------------------------
// 1-Wire Utility functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Get the rom number from the MIB, set in the driver, verify present.
// Use DD_RomData from MIB.
//
// Return: TRUE - device address set and verified present
//         FALSE - no address available or device not present
//
int DeviceSetVerify(int portnum, uchar *buf)
{
   // read the mib entry
   if (ReadFromMIB(DD_RomData, buf, 8) != 8)
      return FALSE;

   // set the ROM number
   owSerialNum(portnum, buf, FALSE);  

   return (owVerify(portnum, FALSE));
}


//----------------------------------------------------------------------
// Read the switch information byte.
//
// Return: TRUE - DS2409 information read and returned in *Info
//         FALSE - error reading DS2409 information
//
//                                                           
int Read2409Info(int portnum, uchar *Info)
{
   uchar send_block[30];
   short send_cnt;

   // access the device 
   if (owAccess(portnum))
   {
      // create a block to send the sets the state
      send_cnt = 0; 
      // read status command 
      send_block[send_cnt++] = 0x5A;
      // control byte
      send_block[send_cnt++] = 0x00;
      // status
      send_block[send_cnt++] = 0xFF;
      // confimation
      send_block[send_cnt++] = 0xFF;

      // now send the block
      if (owBlock(portnum, FALSE, &send_block[0], send_cnt))
      {
         // verify status == confirmation byte
         if (send_block[send_cnt - 1] == send_block[send_cnt - 2]) 
         {
            // record the status and return TRUE
            *Info = send_block[send_cnt - 1];
            return TRUE;
         }
      }
   }

   return FALSE;
}


//--------------------------------------------------------------------------
// Set the latch state of the DS2406.  
// Use DD_ChannelMask, DD_ChannelState from MIB.
//
// Returns: TRUE, success latch state set 
//          FALSE, failure to set latch state
//
int Set2406State(int portnum, uchar *SerialNum)
{
   uchar mask, state;
   short send_cnt=0;
   SwitchProps sw_state;
   int info;

   // read the mib entry for mask and state
   if (ReadFromMIB(DD_ChannelMask, &mask, 1) != 1)
      return FALSE;
   if (ReadFromMIB(DD_ChannelState, &state, 1) != 1)
      return FALSE;

   // check for 0 mask
   if (mask == 0)
      return TRUE;

   // read the current state to effectivily use the mask
   info = (uchar) ReadSwitch12(portnum, FALSE);
   if (info == -1)
      return FALSE;

   sw_state.Chan_B = FALSE;
   sw_state.Chan_A = FALSE;

   if(mask & state & 0x02)
      sw_state.Chan_B = TRUE;
   if(mask & state & 0x01)
      sw_state.Chan_A = TRUE;
   if(mask & info & 0x02)
      sw_state.Chan_B = TRUE;
   if(mask & info & 0x01)
      sw_state.Chan_A = TRUE;

   if(SetSwitch12(portnum, &SerialNum[0], sw_state))
      return TRUE;

   return FALSE;
}


//--------------------------------------------------------------------------
// Set the latch state of the DS2409.  
// Use DD_ChannelMask, DD_ChannelState from MIB.
//
// Returns: TRUE, success latch state set 
//          FALSE, failure to set latch state
//
int Set2409State(int portnum, uchar *SerialNum)
{
   uchar mask, state, info;
   int cmd,xtra;

   // read the mib entry for mask and state
   if (ReadFromMIB(DD_ChannelMask, &mask, 1) != 1)
      return FALSE;
   if (ReadFromMIB(DD_ChannelState, &state, 1) != 1)
      return FALSE;

   // if no channels selected then done
   if ((mask & 0x03) == 0x00)
      return TRUE;

   // decide on command and number of bytes needed
   switch (mask & state & 0x03)
   {
      case 0x00:
         cmd = 0; // ALL_LINES_OFF
         xtra = 1;
         break;
      case 0x02:
         cmd = 2; // smart ON_AUX;
         xtra = 3;
         break;
      case 0x01:  case 0x03:
         cmd = 1; // DIRECT_ON_MAIN;
         xtra = 1;
         break;
   }

   if(SetSwitch1F(portnum, &SerialNum[0], cmd, xtra, &info, FALSE))
      return TRUE;

   return FALSE;
}


//--------------------------------------------------------------------------
// Read the counter value from a DS2423.  
// Use DD_InitState from MIB for counter page.
//
// Returns: TRUE, success, read and placed in *Count 
//          FALSE, failure to read counter
//
int Read2423Counter(int portnum, uchar *SerialNum, ulong *Count)
{
   int rt=FALSE;
   uchar CounterPage;
   short send_cnt=0;

   // read the mib entry for counter page
   if (ReadFromMIB(DD_InitState, &CounterPage, 1) != 1)
      return FALSE;

   // access the device 
   if(ReadCounter(portnum, &SerialNum[0], CounterPage, Count))
      rt = TRUE;

   // return the result flag rt
   return rt;
}

//--------------------------------------------------------------------------
// Find a device and search for a 'TAGD.000' file.  If the file is present
// then read it and place the contents in 'tag_buf' with a max length of
// 'tag_max'.  Use 'DeviceCheckedForTag' function to not repeat file search
// on known devices.
//
// Returns: STATUS_STEP_COMPLETE - search complete (file may have been found) 
//          STATUS_INPROGRESS - not complete, call again
//
int ReadTagNext(int portnum, int *SubStep, uchar *tag_buf, int tag_max)
{
   uchar *retry;
   int length;
   short rslt;
   uchar name[5];

   // check the sub step to see if we need to do a first
   if (*SubStep == 0)
   {
      rslt = owFirst(portnum, TRUE, FALSE);
      *SubStep = 1;
   }
   else
      rslt = owNext(portnum, TRUE, FALSE);

   // if device found 
   if (rslt >= 1)
   {
      // check if we have already check this device for TAG file
      if (!DeviceCheckedForTag(portnum,&retry))
      {
         // attempt to read the TAGD.000 file
         sprintf((char *)&name[0],"TAGD");
         length = owReadFile(portnum, &name[0], &tag_buf[0]);
         if(length >= 0)
         {
            // success, max retry and return complete
            *retry = RETRY_MAX + 1;
            return STATUS_STEP_COMPLETE;
         }
         else
         {         
            // failure, increase retry and return inprogress
            *retry = (uchar)(*retry + 1);
            // clear any junk in tag_buf
            tag_buf[0] = 0; 
            tag_buf[1] = 0; 
            return STATUS_INPROGRESS;
         }
      }
      // already tried this device or not file compatible 
      else
         return STATUS_INPROGRESS;
   }
   // no parts done so complete
   else
      return STATUS_STEP_COMPLETE;
}


//--------------------------------------------------------------------------
// Check if the current device has been check for a 'tag' file.  Return a 
// pointer to the 'retry' number.  This number is used to re-read a
// device several times to before giving up on it having a 'tag' file.
//
// Returns: TRUE, device has already been check for a tag 
//          FALSE, device not yet checked or not exceed max retries 
//
int DeviceCheckedForTag(int portnum, uchar **retry_ptr)
{
   short s,r;
   uchar trom[8];

   // set retry_ptr to null
   *retry_ptr = NULL;

   // read the rom to check
   owSerialNum(portnum, &(uchar)trom[0], TRUE);
   
   // loop through all of the rom's known
   for (s = 0; s < stack_num; s++)
   {
      for (r = 0; r < 8; r++)
         if (rom_stack[s][r] != trom[r])
            break;
      
      // check for a match
      if (r == 8)
      {
         // set pointer to retries in case need
         *retry_ptr = &retry_stack[s];
         // all reties done? 
         return (retry_stack[s] >= RETRY_MAX);
      }
   }

   // not in stack so add it 
   if (stack_num < ROM_STACK_MAX)
   {
      // insert into the stack
      rom_stack[stack_num][0] = 0x00;
      owSerialNum(portnum, &rom_stack[stack_num][0], TRUE);
      // clear the retry
      retry_stack[stack_num] = 0;
      // set pointer to retries in case need
      *retry_ptr = &retry_stack[stack_num];
      // increase size of stack
      stack_num++;
      // could check if this type is even compatible, return TRUE
   }            

   // new or now room to add to stack
   return FALSE;
}

//--------------------------------------------------------------------------
// Set the current device address to the value passed.  Verify if 
// the specified device is currently connected to the 1-Wire Net.
//
// Returns: TRUE, device is present on the 1-Wire Net 
//          FALSE, device NOT present on the 1-Wire Net 
//
int ROMSwitchState(int portnum,uchar *rom)
{
   // set the ROM number
   owSerialNum(portnum,&rom[0],FALSE);

   return owVerify(portnum,FALSE);
}

//--------------------------------------------------------------------------
// General Utility functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Convert the 8 byte arrary to a string representation of a Dallas 1-Wire
// address.  Print least significant byte first.  
//
// Returns: Number of characters added to string not including zero 
//          termination
//
int ROMToStr(char *str, uchar *rom)
{
   int i,scnt=0;
   
   for (i = 7; i >= 0; i--)
      scnt += sprintf(&str[scnt],"%02X",rom[i]);

   return scnt;
}






