//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  1wsensor.c - This utility dynamically reads OBM Tagged
//               1-Wire sensors and read the values.
//
//  Version:   2.00
//  History:   
//

#define SENSOR_SCRIPTS

#include <time.h>
#include <stdlib.h>
#include "1wsensor.h"
#include "msginc.h"
#include "ownet.h"

// external function prototypes 
extern int RunScript(int, Script script[], void *, int);
extern int TAGToString(TAGType *, char *, int);
extern int ParseTAG(int,TAGType *,int,int);
extern int  owAcquire(int,char *,char *);
extern void owRelease(int,char *);
extern int  key_abort(void);

// local function prototypes
int PrintHeader(char *);
int GetLocalTags();
int AddTag(uchar *, int);
void output_status(int, char *);
void CheckTagRevs(void);
void PrintTagInfo(TAGType *);

// globals 
TAGType *TagList[100];
int NumTags=0,VERBOSE=0,TagCount=-1;
FILE *fp;

//----------------------------------------------------------------------
//  This is the Main routine for 1wsensor 
//
int main(short argc, char **argv)
{
   int ShowStatus=FALSE,filenum,first,rslt;
   uchar tagdata[MSGMAX];
   int portnum = 0;
   char return_msg[128];

   // check arguments to see if request instruction with '?' or too many
   if ((argc > 4) || (argc < 2) || ((argc > 1) && (argv[1][0] == '?' || argv[1][1] == '?')))
   {
       printf("\nusage: 1wsensor 1wire_net_port <output_filename> </Verbose>\n"
              "  - auto-sensor read on the specified\n"
              "  - 1-Wire Net port \n" 
              "  - 1wire_net_port - required port name\n"
              "    example: \"COM1\" (Win32 DS2480),\"/dev/cua0\" \n"
              "    (Linux DS2480),\"1\" (Win32 TMEX)\n"
              "  - <output_filename> optional output filename\n"
              "  - </V> optional show verbose output (default not shown)\n"
              "  - press any key to stop auto-sensor scan\n"
              "  - version 2.00\n");
       exit(1);
   }

   // read the default PortNum and PortType 
   if (!owAcquire(portnum, argv[1], return_msg))
   {
       printf("%s",return_msg);
       exit(1);
   }


   // check arguments 
   VERBOSE = FALSE;
   filenum = 0;
   if (argc >= 3)
   {
      if (argv[2][0] != '/')
         filenum = 1;
      else if ((argv[2][1] == 'V') || (argv[2][1] == 'v'))
         VERBOSE = TRUE;

      if (argc == 4)
      {    
         if (argv[3][0] != '/')
            filenum = 2;
         else if ((argv[3][1] == 'V') || (argv[3][1] == 'v'))
            VERBOSE = TRUE;
      }   
   }

   // open the output file  
   fp = NULL;
   if (filenum > 0)
   {
      fp = fopen(argv[filenum],"wb");
      if(fp == NULL)
      {    
         printf("ERROR, Could not open output file!\n");
         exit(1);
      }
      else
         printf("File '%s' opened to write sensor data.\n\n",
                 argv[filenum]);
   }

   // run the script to setup the 1-Wire
   if (!RunScript(portnum, Verify1Wire, NULL, 0))
   { 
      printf("1-Wire Net not successfuly setup, end program\n");
      exit(1);
   }

   // check for local tag
   GetLocalTags();

   // loop to look for new tags and parse existing tags
   do
   {
      // look for tags on the 1-Wire
      if (RunScript(portnum, TagScan, tagdata, MSGMAX))
         AddTag(tagdata, MSGMAX); 

      // loop through each tag
      for (TagCount = 0; (TagCount < NumTags) && !key_abort(); TagCount++)
      {
         // check if this tag should be parsed
         if (TagList[TagCount]->parse_enable)
         {
            // print tag ID and revision
            PrintTagInfo(TagList[TagCount]);
         
            // loop through a partial parse of the tag
            first = TRUE;
            do 
            {
               rslt = ParseTAG(portnum, TagList[TagCount],first,FALSE);
               first = FALSE;
            }
            while ((rslt == NOT_DONE) && (!key_abort()));
         }
      }

      // remove old revision tags
      TagCount=-1;
      CheckTagRevs();   
   }
   while (!key_abort());  

   // close opened file
   if ((fp != NULL) && (fp != stdout))
   {
      printf("File '%s' closed.\n",
              argv[filenum]);
      fclose(fp);
   }

   owRelease(portnum, return_msg);
   printf("%s", return_msg);
   exit(0);

   return 0;
}


//--------------------------------------------------------------------------
//  Prints the current 'tag' information including ID number reversion if 
//  available
//
void PrintTagInfo(TAGType *tag)
{
   char msg[512]; 
   int msgcnt = 0,i;
   time_t tlong;
   struct tm *tstruct;

   // tag ID
   if (tag->tagid[0])
   {
      msgcnt += sprintf(msg + msgcnt," Cluster Num: ");
      for (i = 1; i <= tag->tagid[0]; i++)
         msgcnt += sprintf(msg + msgcnt,"%02X",tag->tagid[i]);
   }
   // (tag revision)
   if (tag->tagrev != -1)
      msgcnt += sprintf(msg + msgcnt,"  Cluster Rev: %d ",tag->tagrev);
   // time/date 
   time(&tlong);
   tstruct = localtime(&tlong);
   msgcnt += sprintf(msg + msgcnt," Time: %02d/%02d/%04d  %02d:%02d:%02d\n",
       tstruct->tm_mon + 1,tstruct->tm_mday,tstruct->tm_year + 1900,   // (1.01)
       tstruct->tm_hour,tstruct->tm_min,tstruct->tm_sec);
   // send output
   output_status(LV_ALWAYS,msg);
}


//--------------------------------------------------------------------------
//  Check the current directory to see   
//
int GetLocalTags(void)
{
   FILE *fp;
   char filename[50];
   uchar buf[MSGMAX];
   int buflen,rslt=0,i;
   char msg[255];

   // loop throuch tagd.0 to tagd.99
   for (i = 0; i < 100; i++)
   {
      // open the input file
      sprintf(filename,"TAGD.%d",i);
      fp = fopen(filename,"rb");
      if(fp == NULL)
         continue;      
      else
      {
         sprintf(msg,"File '%s' opened to read.\n",filename);
         output_status(LV_ALWAYS,msg);
      }

      // read the input file 
      buflen = fread(&buf[0],1,MSGMAX,fp);

      // clean up opened file
      fclose(fp);
      sprintf(msg,"File '%s' closed.\n",filename);
      output_status(LV_ALWAYS,msg);

      // add the tag to the list
      if (AddTag(buf,buflen))
         rslt = TRUE;
   }

   // check if any valid
   if (rslt)
      output_status(LV_ALWAYS,"At least one valid local TAG file found.\n\n");
   
   return rslt;
}


//--------------------------------------------------------------------------
//  Check the current directory to see   
//
int AddTag(uchar *tagdata, int taglen)
{
   char buf[14096];
   int i;

   // sanity check on tag data 
   if ((tagdata[0] == 0) || (tagdata[1] == 0))
      return FALSE;

   // adjust length, loop backward from end to find first non-zero
   for (i = taglen - 1; i >= 0; i--)
      if (tagdata[i] != 0)
      {
         taglen = i + 1;
         break;
      } 
 
   // create the structure to hold the Tag
   TagList[NumTags] = (TAGType *)malloc(sizeof(TAGType) + 1);

   // check if malloc was successfull 
   if (TagList[NumTags] == NULL)
   {
      output_status(LV_ALWAYS,"Insufficient memory to allocate for new Tag.\n"); 
      return FALSE;
   }
   else
      output_status(LV_VERBOSE,"Memory allocated for new Tag.\n");       

   // clear tag ID and revision
   TagList[NumTags]->tagid[0] = 0;
   TagList[NumTags]->tagrev = -1;

   // copy tag msg data to new tag
   TagList[NumTags]->msglen = taglen; 
   for (i = 0; i < taglen; i++)
       TagList[NumTags]->msg[i] = tagdata[i];

   // print out the TAG Tag in human readable form
   if (TAGToString(TagList[NumTags],buf,sizeof(buf)))
   {
      // print data and status
      output_status(LV_VERBOSE,buf);
      output_status(LV_ALWAYS,"Tag is valid.\n\n");
      // mark to parse this tag
      TagList[NumTags]->parse_enable = TRUE;
      // increase the number of tags
      NumTags++;

      return TRUE;
   }
   else
   {
      // error, free memory 
      output_status(LV_ALWAYS,"Tag is not parse-able.\n\n");
      free(TagList[NumTags]);
      return FALSE;
   }      
}

//--------------------------------------------------------------------------
//  Check for duplicate or older revision tags
//
void CheckTagRevs(void)
{
   int i,j,k,refln,tagln;

   // loop through each tag and find and compare revisions
   for (i = 0; i < NumTags; i++)
   {
      // skip if non-parsible or no ID
      if ((!TagList[i]->parse_enable) || (TagList[i]->tagid[0] == 0))
         continue;
      // extract the reference ID length
      refln = TagList[i]->tagid[0];

      // compare all of the tags
      for (j = 0; j < NumTags; j++)
      {
         // make sure not comparing same tag, non-parsible tag, or no ID
         if ((i == j) || (!TagList[j]->parse_enable) || 
                         (TagList[i]->tagid[0] == 0))
            continue;

         // get tag's ID len
         tagln = TagList[j]->tagid[0];

         // different length so not same tag
         if (refln != tagln)   
            continue;
         
         // loop to compare
         for (k = 1; k <= tagln; k++)
            if (TagList[j]->tagid[k] != TagList[i]->tagid[k])
               break;
         
         // check loop
         if (k == (tagln + 1))
         {
            // same ID so now check revision
            if (TagList[j]->tagrev == TagList[i]->tagrev)
               TagList[j]->parse_enable = FALSE; // duplicate
            else if (TagList[j]->tagrev < TagList[i]->tagrev)                              
               TagList[j]->parse_enable = FALSE; // old rev
            else if (TagList[j]->tagrev > TagList[i]->tagrev)
            {
               // newer rev so kill off the reference and break out of loop
               TagList[i]->parse_enable = FALSE;
               break;   
            }                              
         }
      }
   }
}

//--------------------------------------------------------------------------
//  output status message
//
void output_status(int level, char *st)
{
   // skip null strings
   if (st[0] == 0)
      return;

   // check if in verbose mode
   if ((level >= LV_ALWAYS) || VERBOSE)
   { 
      // print header for each line (give tag number)
      if (TagCount >= 0)
      {
         printf("%d|%s",TagCount,st);
         if(fp != NULL)
            fprintf(fp,"%d|%s",TagCount,st);
      }
      else
      {
         printf("-|%s",st);
         if(fp != NULL)
            fprintf(fp,"-|%s",st);
      }
   }
}


